﻿/**********************************
* @file isp.c
* @brief
* @author
* @version V0.0.0
* @date 2020.02.20
* @details 
* @copyright
* @note 
**********************************/

/* Includes ------------------------------------------------------------------*/

#include "CSU38F20.h"
#include "isp.h"

/* Fuction Declare ----------------------------------------------------------*/

/**********************************
* @fn  void write_flash(unsigned int write_address_flash[],unsigned int write_data_flash[],unsigned char write_flash_number)
* @brief write data to flash
* @param[out] none.
* @param[in] address_flash[],write_data_flash[],write_flash_number.
* @return none.
**********************************/
void write_flash(unsigned int write_address_flash[],unsigned int write_data_flash[],unsigned char write_flash_number);

/**********************************
* @fn  void change_data_flash(unsigned int change_address_flash[],unsigned int change_data_flash[])
* @brief change data to flash
* @param[out] none.
* @param[in] change_address_flash[],change_data_flash[].
* @return none.
**********************************/
void change_data_flash(unsigned int change_address_flash[],unsigned int change_data_flash[]);

/**********************************
* @fn  unsigned int read_flash(unsigned int read_address_flash)
* @brief read flash
* @param[out].
* @param[in] read_address_flash.
* @return read_data_flash.
**********************************/
unsigned int read_flash(unsigned int read_address_flash);

/**********************************
* @fn  static void copy_flash_to_row0(unsigned int copy_address_flash)
* @brief copy flash to row0
* @param[out] none.
* @param[in] copy_address_flash.
* @return none.
**********************************/
static void copy_flash_to_row0(unsigned int copy_address_flash);

/**********************************
* @fn  static void write_page_latch(unsigned int write_address_page_latch,unsigned int write_data_page_latch)
* @brief write data to page latch
* @param[out] none.
* @param[in] write_address_page_latch,write_data_page_latch.
* @return none.
**********************************/
static void write_page_latch(unsigned int write_address_page_latch,unsigned int write_data_page_latch);

/* Fuction Define -----------------------------------------------------------*/

/**********************************
* @fn  unsigned int read_flash(unsigned int read_address_flash)
* @brief read flash
* @param[out].
* @param[in] read_address_flash.
* @return read_data_flash.
**********************************/
unsigned int read_flash(unsigned int read_address_flash)
{
	unsigned int read_data_flash = 0;
	GIE = 0;		//关总中断
	EADRL = read_address_flash;			//赋读取地址
	EADRH = read_address_flash >> 8;
	asm("MOVP");			//清除Page latch
	asm("nop");
	read_data_flash = (EDATH << 8) | WORK;	//返回16位数据
	GIE = 1;		//开总中断，读操作结束
	asm("nop");
	return read_data_flash;
}

/**********************************
* @fn  void write_flash(unsigned int write_address_flash[],unsigned int write_data_flash[],unsigned char write_flash_number)
* @brief write data to flash
* @param[out] none.
* @param[in] address_flash[],write_data_flash[],write_flash_number.
* @return none.
**********************************/
void write_flash(unsigned int write_address_flash[],unsigned int write_data_flash[],unsigned char write_flash_number)
{
	unsigned char write_flash_counter = 0;
	GIE = 0;	//关总中断
	
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 5AH");			//清除Page latch
	asm("nop");
	
	EADRL = write_address_flash[0];			//赋地址，页擦除模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = write_address_flash[0] >> 8;
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 0A5H");			//页擦除
	asm("nop");
	
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 5AH");			//清除Page latch
	asm("nop");
	
	do{
		write_page_latch(write_address_flash[write_flash_counter],write_data_flash[write_flash_counter]);
	}while(++write_flash_counter < write_flash_number);
	
	EADRL = write_address_flash[0];			//赋地址，页烧录模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = write_address_flash[0] >> 8;
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 96H");			//页烧录
	asm("nop");
	
	GIE = 1;	//开总中断
}

/**********************************
* @fn  void change_data_flash(unsigned int change_address_flash[],unsigned int change_data_flash[])
* @brief change data to flash
* @param[out] none.
* @param[in] change_address_flash[],change_data_flash[].
* @return none.
**********************************/
void change_data_flash(unsigned int change_address_flash[],unsigned int change_data_flash[])
{
	unsigned char change_flash_counter = 0;
	unsigned char write_flash_counter = 0;
	unsigned int change_address_temp = 0;
	unsigned int change_data_temp = 0;
	GIE = 0;	//关总中断
	change_address_temp = change_address_flash[0] & 0xFFE0;
	copy_flash_to_row0(change_address_temp);	//将待操作片区内的数据缓存到row0区域
	
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 5AH");			//清除Page latch
	asm("nop");
	
	EADRL = change_address_temp;	//赋地址，页擦除模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = change_address_temp >> 8;
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 0A5H");			//页擦除
	asm("nop");
	
	do{
		if(change_address_temp == change_address_flash[change_flash_counter])	//若写入地址处的数据需要修改，则替换为修改后的数据，否则从缓存区读取数据
		{
			change_data_temp = change_data_flash[change_flash_counter];
			change_flash_counter++;
		}
		else
		{
			EADRL = write_flash_counter;			//赋读取地址
			EADRH = 0x20;
			asm("MOVP");			//读数据
			asm("nop");
			change_data_temp = (EDATH << 8) | WORK;	//返回16位数据
			asm("nop");
		}	
		write_page_latch(change_address_temp,change_data_temp);
		change_address_temp++;
	}while(++write_flash_counter <= 0x1F);
	
	EADRL = change_address_flash[0];			//赋地址，页烧录模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = change_address_flash[0] >> 8;
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 96H");			//页烧录
	asm("nop");
//	
	GIE = 1;	//开总中断
}

/**********************************
* @fn  static void copy_flash_to_row0(unsigned int copy_address_flash)
* @brief copy flash to row0
* @param[out] none.
* @param[in] copy_address_flash.
* @return none.
**********************************/
static void copy_flash_to_row0(unsigned int copy_address_flash)
{
	unsigned char copy_flash_counter = 0;
	unsigned int copy_flash_data = 0;
	
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 5AH");			//清除Page latch
	asm("nop");
	
	EADRL = 0x00;			//擦除缓存区，页擦除模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = 0x20;
	WRPRT = 0x96;			//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 0A5H");		//页擦除
	asm("nop");
	
	WRPRT = 0x96;				//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 5AH");			//清除Page latch
	asm("nop");
	
	copy_address_flash &= 0xFFE0;
	for(copy_flash_counter=0;copy_flash_counter<=0x1F;copy_flash_counter++)
	{
		EADRL = copy_address_flash | copy_flash_counter;			//赋读取地址
		EADRH = copy_address_flash >> 8;
		asm("MOVP");			//读数据
		asm("nop");
		copy_flash_data = (EDATH << 8) | WORK;	//返回16位数据
		asm("nop");
		
		EDATH = copy_flash_data >> 8;		//赋数据高8位
		WRPRT = 0x96;						//解锁TBLP操作
		WRPRT = 0x69;
		WRPRT = 0x5A;
		WORK = copy_flash_data;				//赋数据低8位
		asm("TBLP 69H");					//将数据按地址写入Page latch
		asm("nop");
	}
	
	EADRL = 0x00;			//将数据复制至缓存区，页烧录模式下仅EDAR[15:5]有效，而EDAR[4:0]无效
	EADRH = 0x20;
	WRPRT = 0x96;			//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	asm("TBLP 96H");		//页烧录
	asm("nop");
}


/**********************************
* @fn  static void write_page_latch(unsigned int write_address_page_latch,unsigned int write_data_page_latch)
* @brief write data to page latch
* @param[out] none.
* @param[in] write_address_page_latch,write_data_page_latch.
* @return none.
**********************************/
static void write_page_latch(unsigned int write_address_page_latch,unsigned int write_data_page_latch)
{
	EADRL = write_address_page_latch;			//赋地址，Page latch write模式下仅EDAR[4:0]有效，而EDAR[15:5]无效
	EADRH = write_address_page_latch >> 8;
	EDATH = write_data_page_latch >> 8;			//赋数据高8位
	WRPRT = 0x96;								//解锁TBLP操作
	WRPRT = 0x69;
	WRPRT = 0x5A;
	WORK = write_data_page_latch;				//赋数据低8位
	asm("TBLP 69H");							//将数据按地址写入Page latch
	asm("nop");
}